﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapContactToCrm : MapperRelatedBase
    {
        public MapContactToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<Contact> MapInsert(PersonContact contact, Account provider)
        {
            // map entity
            var entity = new Contact
            {
                Id = Guid.NewGuid(),
                FirstName = contact.FirstName,
                MiddleName = contact.MiddleName,
                LastName = contact.LastName
            };

            if (contact.IsVeteranSpecified)
            {
                entity.ppms_isveteran = contact.IsVeteran;
            }

            if (contact.EffectiveDateSpecified && contact.EffectiveDate.Year > EnumHelper.MinYear)
            {
                entity.ppms_effectivedate = contact.EffectiveDate;
            }

            await Task.Run(() => {});

            return entity;
        }

        public async Task<Contact> MapUpdate(PersonContact entity, Account provider)
        {
            Contact crmEntity = null;
            // get entity, if CorrelationId provided
            if (!string.IsNullOrEmpty(entity.CorrelationId))
            {
                crmEntity = GetCrmEntity(provider, entity.CorrelationId);
            }

            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            var newEntity = new Contact()
            {
                Id = crmEntity.Id,
                FirstName = crmEntity.FirstName,
                MiddleName = crmEntity.MiddleName,
                LastName = crmEntity.LastName,
                ppms_effectivedate = crmEntity.ppms_effectivedate
            };

            // Map fields
            if (IsChanged(entity.FirstName, newEntity.FirstName))
                newEntity.FirstName = entity.FirstName;

            if (IsChanged(entity.MiddleName, newEntity.MiddleName))
                newEntity.MiddleName = entity.MiddleName;

            if (IsChanged(entity.LastName, newEntity.LastName))
                newEntity.LastName = entity.LastName;

            if (entity.IsVeteranSpecified)
                newEntity.ppms_isveteran = entity.IsVeteran;

            if (entity.EffectiveDateSpecified && entity.EffectiveDate.Year > EnumHelper.MinYear)
                newEntity.ppms_effectivedate = entity.EffectiveDate;

            return newEntity;
        }

        public SetStateRequest MapDelete(Contact crmItem)
        {
            if (crmItem != null)
            {
                return DeactivateEntity(crmItem);
            }
            return null;
        }

        private static Contact GetCrmEntity(Account provider, string id)
        {
            var list = provider.contact_customer_accounts;
            if (list == null) return null;

            var guid = new Guid(id);
            var contacts = list as Contact[] ?? list.ToArray();

            return contacts.Any() ? contacts.FirstOrDefault(x => x.Id == guid) : null;
        }

        private static PersonContact ConvertEntity<T>(T entity)
        {
            return (PersonContact)Convert.ChangeType(entity, typeof(PersonContact));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "contact_customer_accounts");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null)
                        account.contact_customer_accounts = ConvertEntityList<Contact>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<T>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var entity1 = entity;
                var matches = provider.contact_customer_accounts.Where(p => p.Id == new Guid(entity1.CorrelationId));
                list.AddRange((IEnumerable<T>)matches);
            }

            return EntityDelete((IEnumerable<Contact>)list);
        }
    }
}